const { DB } = require("../../db");
const { sendEmail } = require("../../helper/sendEmail");
const { sendToUser } = require("../../middleware/sendToUser");
const Router = require("express").Router();
const { verifyAdmin } = require("../../middleware/verifyAdmin");
const { verifySession } = require("../../middleware/verifySession");
const joi = require("joi");
const mongoose = require("mongoose");
const { getWss } = require("../../websockets/websocket");

//get users bookmark markets
Router.get("/bookmarks/:page/:limit", verifySession, async (req, res) => {
  try {
    const page = parseInt(req.params.page) || 1;
    const limit = parseInt(req.params?.limit) || 26;

    const total = await DB.BOOKMARK.countDocuments({ uid: req.user._id });
    const total_pages = Math.ceil(total / limit);

    const bookmarks = await DB.BOOKMARK.find({ uid: req.user._id })
      .populate("mid", "-img")
      .skip((page - 1) * limit)
      .limit(limit + 1)
      .sort({ createdAt: -1 });

    const hasMore = bookmarks.length > limit;
    if (hasMore) bookmarks.pop(); // remove the extra item

    res.status(201).json({
      data: {
        data: bookmarks,
        hasMore,
        total_pages,
      },
    });
  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request",
    });
  }
});

// user bookmark market
Router.put(
  "/toggle/bookmark/market/:marketID",
  verifySession,
  async (req, res) => {
    try {
      // check market
      const find_market = await DB.MARKET.findById(req.params.marketID);

      if (!find_market)
        throw new Error("Unable to complete request, Can't find market");

      //   now check if the bookmark already exits
      const find_bookmark = await DB.BOOKMARK.findOne({
        uid: req.user._id,
        mid: find_market._id,
      });

      //   create bookmark
      if (!find_bookmark) {
        const add_bookmark = await DB.BOOKMARK.create({
          uid: req.user._id,
          mid: find_market._id,
        });

        sendToUser(req.user._id?.toString(), {
          type: "ACCOUNT_NOTIFICATION",
          payload: {
            type: "info",
            body: {
              title: "Bookmark",
              text: `A new market bookmark has been added to your collection.`,
            },
          },
        });

        await DB.NOTIFICATION.create({
          uid: req.user._id,
          type: "info",
          body: {
            title: "Bookmark",
            text: `A new market bookmark has been added to your collection.`,
          },
        });

        res.status(201).json({
          data: "bookmark created",
        });
        return;
      }

      await DB.BOOKMARK.findByIdAndDelete(find_bookmark._id);
      res.status(201).json({
        data: "deleted bookmark",
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request",
      });
    }
  }
);

const COMMENT_ = joi
  .object({
    content: joi.string().required(),
    replying_to: joi.string(),
  })
  .strict();

// user comment for market
Router.post("/comment/market/:marketID", verifySession, async (req, res) => {
  try {
    let { error, value } = COMMENT_.validate(req.body);
    if (error) throw new Error(error.details[0].message);

    let { content, replying_to } = value;

    // check market
    const find_market = await DB.MARKET.findById(req.params.marketID);

    if (!find_market)
      throw new Error("Unable to complete request, Can't find market");

    if (
      replying_to?.length > 0 &&
      !mongoose.Types.ObjectId.isValid(replying_to)
    )
      throw new Error("replying_to should be a valid object ID");

    if (replying_to) {
      // find replying_to and the user for notification
      const getreplying_to = await DB.COMMENT.findById(replying_to).populate(
        "mid",
        "title description category"
      );

      if (!getreplying_to)
        throw new Error("Can't find comment with the replyID");

      sendToUser(getreplying_to.uid?.toString(), {
        type: "ACCOUNT_NOTIFICATION",
        payload: {
          type: "info",
          body: {
            title: "Comment",
            text: `${req.user.username} replied to your comment for the market ${getreplying_to.mid.title}`,
          },
        },
      });

      await DB.NOTIFICATION.create({
        uid: getreplying_to.uid,
        type: "info",
        body: {
          title: "Comment",
          text: `${req.user.username} replied to your comment for the market ${getreplying_to.mid.title}`,
        },
      });
    }

    // create comment
    const create = await DB.COMMENT.create({
      uid: req.user._id,
      mid: find_market._id,
      content,
      replying_to,
    });
    res.status(201).json({
      data: create,
    });
  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request",
    });
  }
});

// user get comments for any market
Router.get("/comment/market/:marketID/:page/:limit", async (req, res) => {
  try {
    // check market
    const find_market = await DB.MARKET.findById(req.params.marketID);

    if (!find_market)
      throw new Error("Unable to complete request, Can't find market");

    const page = parseInt(req.params.page) || 1;
    const limit = parseInt(req.params?.limit) || 26;

    const total = await DB.COMMENT.countDocuments({ mid: find_market._id });
    const total_pages = Math.ceil(total / limit);

    const every_ = await DB.COMMENT.find({ mid: find_market._id })
      .populate("uid", "image email username name phone")
      .populate("replying_to")
      .skip((page - 1) * limit)
      .limit(limit + 1)
      .sort({ createdAt: -1 });

    const hasMore = every_.length > limit;
    if (hasMore) every_.pop(); // remove the extra item

    res.status(201).json({
      data: {
        data: every_,
        hasMore,
        total_pages,
      },
    });
  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request",
    });
  }
});

const REACTIONS_ = joi
  .object({
    reactions: joi.string().required(),
  })
  .strict();

// user get comments for any market
Router.put("/react/:commentID", verifySession, async (req, res) => {
  try {
    let { error, value } = REACTIONS_.validate(req.body);
    if (error) throw new Error(error.details[0].message);

    let { reactions } = value;

    // check market
    const find_comment = await DB.COMMENT.findById(
      req.params.commentID
    ).populate("mid", "title description category");

    if (!find_comment)
      throw new Error("Unable to complete request, Can't find comment");
    const userId = req.user._id; // assuming you have req.user set by auth middleware

    // Check if user already reacted with the same reaction
    const existingReactionIndex = find_comment.reactions.findIndex(
      (r) => r.uid.toString() === userId.toString()
    );

    if (existingReactionIndex > -1) {
      // ✅ User already reacted — remove the reaction
      find_comment.reactions.splice(existingReactionIndex, 1);
    } else {
      sendToUser(find_comment.uid?.toString(), {
        type: "ACCOUNT_NOTIFICATION",
        payload: {
          type: "info",
          body: {
            title: "Reaction",
            text: `${req.user.username} reacted to your comment for the market ${find_comment.mid.title}`,
          },
        },
      });

      await DB.NOTIFICATION.create({
        uid: find_comment.uid,
        type: "info",
        body: {
          title: "Reaction",
          text: `${req.user.username} reacted to your comment for the market ${find_comment.mid.title}`,
        },
      });

      // ✅ Add new reaction
      find_comment.reactions.push({ uid: userId, reaction: reactions });
    }

    await find_comment.save();
    res.status(200).json({
      data: find_comment,
    });
  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request",
    });
  }
});

const ENTER_MAR_ = joi
  .object({
    amount: joi.number().required(),
    predicting: joi.string().valid("yes", "no").required(),
  })
  .strict();

// user entering any market
Router.post("/enter/:marketID", verifySession, async (req, res) => {
  try {
    let { error, value } = ENTER_MAR_.validate(req.body);
    if (error) throw new Error(error.details[0].message);

    let { amount, predicting } = value;

    // check if market exists
    const find_market = await DB.MARKET.findById(req.params.marketID);

    if (!find_market)
      throw new Error("Unable to complete request, Can't find market");

    if (amount < find_market.min)
      throw new Error(
        `Unable to complete request. The minimum amount required to enter a trade is ${find_market.min}`
      );

    if (amount > find_market.max)
      throw new Error(
        `Unable to complete request. The maximum amount required to enter a trade is ${find_market.max}`
      );

    // fess check there amount, the wallet amount to see if the have the amount the are going to enter market with
    const user_wallet = await DB.WALLET.findOne({
      uid: req.user._id,
      is_demo: req.user.is_demo,
    });

    if (!user_wallet)
      throw new Error(
        "Unable to predict market, fund your user wallet to continue"
      );

    // now calc user_wallet.usdt_balance for the user
    const calc = user_wallet.usdt_balance - user_wallet.on_hold;
    if (calc <= amount)
      throw new Error("Unable to predict market, Insufficient wallet balance");

    // now remove the amount from the user wallet
    const remov = await DB.WALLET.findByIdAndUpdate(
      user_wallet._id,
      {
        $inc: { on_hold: amount, pending: amount },
      },
      { new: true }
    );

    // now predict here
    const predict = await DB.PREDICTION.create({
      uid: req.user._id,
      mid: find_market._id,
      predict: predicting,
      is_demo: req.user.is_demo,
      amount,
    });

    // now update the total_stakes
    if (req.user.is_demo) {
      await DB.USER.findByIdAndUpdate(
        req.user._id,
        { $inc: { total_demo_stakes: 1, total_demo_prediction: amount } },
        { new: true }
      );
    } else {
      await DB.USER.findByIdAndUpdate(
        req.user._id,
        { $inc: { total_stakes: 1, total_prediction: amount } },
        { new: true }
      );
    }

    res.status(200).json({
      data: predict,
    });

    const uid = req.query.uid; // 👈 you can pass ?uid=USER_ID from frontend

    const id = find_market._id;

    const markets = await DB.MARKET.aggregate([
      { $match: { _id: new mongoose.Types.ObjectId(id) } },

      // Lookup predictions
      {
        $lookup: {
          from: "predictions",
          localField: "_id",
          foreignField: "mid",
          as: "predictions",
        },
      },

      // Add computed stats
      {
        $addFields: {
          total_predictions: { $size: "$predictions" },
          yes_predictions: {
            $size: {
              $filter: {
                input: "$predictions",
                as: "p",
                cond: { $eq: ["$$p.predict", "yes"] },
              },
            },
          },
          no_predictions: {
            $size: {
              $filter: {
                input: "$predictions",
                as: "p",
                cond: { $eq: ["$$p.predict", "no"] },
              },
            },
          },
          total_amount: { $sum: "$predictions.amount" },
          yes_amount: {
            $sum: {
              $map: {
                input: {
                  $filter: {
                    input: "$predictions",
                    as: "p",
                    cond: { $eq: ["$$p.predict", "yes"] },
                  },
                },
                as: "p",
                in: "$$p.amount",
              },
            },
          },
          no_amount: {
            $sum: {
              $map: {
                input: {
                  $filter: {
                    input: "$predictions",
                    as: "p",
                    cond: { $eq: ["$$p.predict", "no"] },
                  },
                },
                as: "p",
                in: "$$p.amount",
              },
            },
          },

          // 👇 NEW: Check if this user has already predicted
          hasPredicted: {
            $in: [
              uid ? new mongoose.Types.ObjectId(uid) : null,
              "$predictions.uid",
            ],
          },
        },
      },

      // Hide raw predictions data
      {
        $project: {
          predictions: 0,
          img: 0,
        },
      },
    ]);

    const wss = getWss(); // safely get current instance

    // 🔥 BROADCAST TO ALL WEBSOCKET CLIENTS
    wss.clients.forEach((client) => {
      if (client.readyState === 1) {
        client.send(
          JSON.stringify({
            type: "NEW_PREDICTION",
            payload: markets,
          })
        );
      }
    });
  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request",
    });
  }
});

async function getUserTotals(userId) {
  const result = await DB.PREDICTION.aggregate([
    {
      $match: {
        uid: new mongoose.Types.ObjectId(userId),
        predict: { $in: ["yes", "no"] },
      },
    },
    {
      $group: {
        _id: { predict: "$predict", is_demo: "$is_demo" },
        totalAmount: { $sum: "$amount" },
      },
    },
    {
      $group: {
        _id: "$_id.predict",
        accounts: {
          $push: {
            is_demo: "$_id.is_demo",
            totalAmount: "$totalAmount",
          },
        },
      },
    },
    {
      $project: {
        _id: 0,
        predict: "$_id",
        demoAccount: {
          $sum: {
            $map: {
              input: {
                $filter: {
                  input: "$accounts",
                  as: "acc",
                  cond: { $eq: ["$$acc.is_demo", true] },
                },
              },
              as: "item",
              in: "$$item.totalAmount",
            },
          },
        },
        realAccount: {
          $sum: {
            $map: {
              input: {
                $filter: {
                  input: "$accounts",
                  as: "acc",
                  cond: { $eq: ["$$acc.is_demo", false] },
                },
              },
              as: "item",
              in: "$$item.totalAmount",
            },
          },
        },
      },
    },
  ]);

  // Convert array → desired output shape
  const response = {
    totalYes: { demoAccount: 0, realAccount: 0 },
    totalNo: { demoAccount: 0, realAccount: 0 },
  };

  for (const row of result) {
    if (row.predict === "yes") {
      response.totalYes.demoAccount = row.demoAccount || 0;
      response.totalYes.realAccount = row.realAccount || 0;
    } else if (row.predict === "no") {
      response.totalNo.demoAccount = row.demoAccount || 0;
      response.totalNo.realAccount = row.realAccount || 0;
    }
  }

  return response;
}

// get prediction market history
Router.get(
  "/prediction/history/:page/:limit",
  verifySession,
  async (req, res) => {
    try {
      const page = parseInt(req.params.page) || 1;
      const limit = parseInt(req.params?.limit) || 26;

      const total = await DB.PREDICTION.countDocuments({ uid: req.user._id });
      const total_pages = Math.ceil(total / limit);

      const ttals = await getUserTotals(req.user._id);

      const every_ = await DB.PREDICTION.find({
        uid: req.user._id,
        is_demo: req.user.is_demo,
      })
        .populate("mid", "-img")
        .skip((page - 1) * limit)
        .limit(limit + 1)
        .sort({ createdAt: -1 });

      const hasMore = every_.length > limit;
      if (hasMore) every_.pop(); // remove the extra item

      res.status(201).json({
        data: {
          data: every_,
          hasMore,
          total_pages,
        },
        metrics: {
          total_stakes: req.user.is_demo
            ? req.user.total_demo_stakes
            : req.user.total_stakes,
          win_rate: ttals,
          total_winning: req.user.is_demo
            ? req.user.total_demo_winning
            : req.user.total_winning,
          total_predictions: req.user.is_demo
            ? req.user.total_demo_prediction
            : req.user.total_prediction,
        },
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request",
      });
    }
  }
);

module.exports = Router;
