const { DB } = require("../../db");
const { sendToUser } = require("../../middleware/sendToUser");
const Router = require("express").Router();
const { verifyAdmin } = require("../../middleware/verifyAdmin");
const { verifySession } = require("../../middleware/verifySession");
const joi = require("joi");

// admin get Wallet Requests
Router.get("/wallet/requests/:page/:limit", verifySession, async (req, res) => {
  try {
    const page = parseInt(req.params.page) || 1;
    const limit = parseInt(req.params?.limit) || 26;

    const total = await DB.DE_WITH.countDocuments();
    const total_pages = Math.ceil(total / limit);

    const every_ = await DB.DE_WITH.find()
      .select("-img")
      .populate("uid", "image email username name")
      .skip((page - 1) * limit)
      .limit(limit + 1)
      .sort({ createdAt: -1 });

    const hasMore = every_.length > limit;
    if (hasMore) every_.pop(); // remove the extra item

    const total_pending_deposits = await DB.DE_WITH.countDocuments({
      type: "deposit",
      status: "pending",
    });
    const total_pending_withdrawal = await DB.DE_WITH.countDocuments({
      type: "withdrawal",
      status: "pending",
    });
    const total_cancelled_request = await DB.DE_WITH.countDocuments({
      status: "cancelled",
    });

    res.status(201).json({
      data: {
        metrics: {
          total_requests: total,
          total_pending_deposits,
          total_pending_withdrawal,
          total_cancelled_request,
        },
        data: every_,
        hasMore,
        total_pages,
      },
    });
  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request",
    });
  }
});

const TOGGLE_TRX = joi
  .object({
    type: joi.string().valid("approve", "cancel").required(),
    amount: joi.number(),
  })
  .strict();

// admin approving or cancelling any tranx either deposit or withdrawal
Router.put("/process/trnx/:id", verifySession, async (req, res) => {
  try {
    let { error, value } = TOGGLE_TRX.validate(req.body);
    if (error) throw new Error(error.details[0].message);
    //
    let { type, amount } = value;

    // first find the trnx
    const find_trnx = await DB.DE_WITH.findById(req.params.id);

    if (!find_trnx) throw new Error("Can't find transaction");

    const findUser = await DB.USER.findById(find_trnx.uid);
    if (!findUser) throw new Error("Can't find user");

    if (find_trnx.status !== "pending")
      throw new Error(
        `Unable to complete request, this transaction has already been ${find_trnx.status}`
      );

    // cancel transaction here!
    if (type == "cancel") {
      // update status to cancel and save
      const upda = await DB.DE_WITH.findByIdAndUpdate(
        find_trnx._id,
        { status: "cancelled" },
        { new: true }
      );

      sendToUser(upda.uid?.toString(), {
        type: "ACCOUNT_NOTIFICATION",
        payload: {
          type: "alert",
          body: {
            title: "Account TOP UP",
            text: `Your top-up was unsuccessful, so it has been cancelled.`,
          },
        },
      });

      await DB.NOTIFICATION.create({
        uid: upda.uid,
        type: "alert",
        body: {
          title: "Account TOP UP",
          text: `Your top-up was unsuccessful, so it has been cancelled.`,
        },
      });

      res.status(201).json({
        data: upda,
      });
      return;
    }

    if (!amount) throw new Error("Amount is required to approve a transaction");

    // approving the transaction =>> update the user balance and if no balance, then create one.
    // check if the user has a wallet model
    let user_wallet;
    user_wallet = await DB.WALLET.findOne({
      uid: findUser._id,
      is_demo: findUser.is_demo,
    });

    if (!user_wallet) {
      user_wallet = await DB.WALLET.create({
        is_demo: findUser.is_demo,
        uid: findUser._id,
      });
    }

    // now add the amount from the admin to the user wallet balance
    await DB.WALLET.findByIdAndUpdate(
      user_wallet._id,
      {
        $inc: { usdt_balance: amount },
      },
      { new: true }
    );

    const updaa = await DB.DE_WITH.findByIdAndUpdate(
      find_trnx._id,
      { status: "completed" },
      { new: true }
    );

    sendToUser(find_trnx.uid?.toString(), {
      type: "ACCOUNT_NOTIFICATION",
      payload: {
        type: "success",
        body: {
          title: "Account TOP UP",
          text: `Your top-up has been successfully completed.`,
        },
      },
    });

    await DB.NOTIFICATION.create({
      uid: find_trnx.uid,
      type: "success",
      body: {
        title: "Account TOP UP",
        text: `Your top-up has been successfully completed.`,
      },
    });

    res.status(201).json({
      data: updaa,
    });
  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request",
    });
  }
});

module.exports = Router;
