const Router = require("express").Router();
const { DB } = require("../../db");
const { verifyAdmin } = require("../../middleware/verifyAdmin");
const { verifySession } = require("../../middleware/verifySession");
const joi = require("joi");
var multipart = require("connect-multiparty");
var multipartMiddleware = multipart();
var cloudinary = require("cloudinary").v2;
const lowerCase = require("../../middleware/convert_lowercase");
const moment = require("moment");
const mongoose = require("mongoose");
const { APP_PERCENTAGE } = require("../../helper/constant");
const { getWss } = require("../../websockets/websocket");
const { sendToUser } = require("../../middleware/sendToUser");

cloudinary.config({
  cloud_name: "dqdndojli",
  api_key: "517252993479136",
  api_secret: "uX_gbq1gKHMO8Bs9txUTOjs1ecM",
  secure: true,
});

const create_market = joi
  .object({
    end_date: joi.string().required(),
    title: joi.string().required(),
    description: joi.string().required(),
    category: joi.string().required(),
    min: joi.string().required(),
    max: joi.string().required(),
    rules: joi.string(),
    has_seed: joi.string().valid("true", "false").required(),
    yes_seed: joi.string(),
    no_seed: joi.string(),
  })
  .strict();

// CREATE MARKET
Router.post(
  "/create/market",
  verifySession,
  verifyAdmin,
  multipartMiddleware,
  lowerCase,
  async (req, res) => {
    try {
      let { error, value } = create_market.validate(req.body);
      if (error) throw new Error(error.details[0].message);

      const {
        end_date,
        title,
        description,
        category,
        min,
        max,
        rules,
        has_seed,
        yes_seed,
        no_seed,
      } = value;

      if (!moment(end_date).isValid())
        throw new Error('"end_date" must be a valid date format');

      // Optionally ensure it’s not in the past, etc.
      if (moment(end_date).isBefore(moment(), "day"))
        throw new Error('"end_date" cannot be in the past');

      if (!req?.files?.photo?.path)
        throw new Error("photo is required for creating any market");

      // upload the image here
      const uploadingData = await cloudinary.uploader.upload(
        req?.files?.photo?.path
      );

      let newUser = await DB.MARKET.create({
        end_date,
        title,
        description,
        category,
        min,
        max,
        rules,
        image: uploadingData.url,
        img: uploadingData,
        has_seed: has_seed == "true" ? true : false,
        yes_seed,
        no_seed,
      });

      if (has_seed == "true") {
        // now predict here if there yes seed and no seed
        const yes_predict_seed = await DB.PREDICTION.create({
          is_seed: true,
          mid: newUser._id,
          predict: "yes",
          amount: yes_seed,
          is_demo: false,
        });

        const no_predict_seed = await DB.PREDICTION.create({
          is_seed: true,
          mid: newUser._id,
          predict: "no",
          amount: no_seed,
          is_demo: false,
        });

        const yes_fake_predict_seed = await DB.PREDICTION.create({
          is_seed: true,
          mid: newUser._id,
          predict: "yes",
          amount: yes_seed,
          is_demo: true,
        });

        const no_fake_predict_seed = await DB.PREDICTION.create({
          is_seed: true,
          mid: newUser._id,
          predict: "no",
          amount: no_seed,
          is_demo: true,
        });
      }

      const wss = getWss(); // safely get current instance

      // 🔥 BROADCAST TO ALL WEBSOCKET CLIENTS
      wss.clients.forEach((client) => {
        if (client.readyState === 1) {
          client.send(
            JSON.stringify({
              type: "NEW_MARKET",
              payload: {
                data: { ...newUser._doc, img: null },
              },
            })
          );
        }
      });

      res.status(201).json({
        data: { ...newUser._doc, img: null },
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request",
      });
    }
  }
);

// EDIT MARKET
Router.put(
  "/edit/market/:id",
  verifySession,
  verifyAdmin,
  multipartMiddleware,
  lowerCase,
  async (req, res) => {
    try {
      let { error, value } = create_market.validate(req.body);
      if (error) throw new Error(error.details[0].message);

      const { end_date, title, description, category, min, max, rules } = value;

      const find_market = await DB.MARKET.findById(req.params.id);

      if (!find_market)
        throw new Error("Unable to complete request, Can't find market");

      if (!moment(end_date).isValid())
        throw new Error('"end_date" must be a valid date format');

      // Optionally ensure it’s not in the past, etc.
      if (moment(end_date).isBefore(moment(), "day"))
        throw new Error('"end_date" cannot be in the past');

      const wss = getWss(); // safely get current instance

      if (!req?.files?.photo?.path) {
        // edit other stuffs
        let newUser = await DB.MARKET.findByIdAndUpdate(
          find_market._id,
          {
            end_date,
            title,
            description,
            category,
            min,
            max,
            rules,
          },
          { new: true }
        );

        // 🔥 BROADCAST TO ALL WEBSOCKET CLIENTS
        wss.clients.forEach((client) => {
          if (client.readyState === 1) {
            client.send(
              JSON.stringify({
                type: "EDITED_MARKET",
                payload: {
                  data: { ...newUser._doc, img: null },
                },
              })
            );
          }
        });

        res.status(201).json({
          data: { ...newUser._doc, img: null },
        });
        return;
      }

      //   delete the old image here and store the new one
      if (find_market?.img?.public_id) {
        // delete the image HERE
        const deleuploadingData = await cloudinary.uploader.destroy(
          find_market?.img?.public_id
        );

        console.log("uploadingData =>>> ", deleuploadingData);
      }

      // upload the image here
      const uploadingData = await cloudinary.uploader.upload(
        req?.files?.photo?.path
      );

      let newUser = await DB.MARKET.findByIdAndUpdate(
        find_market._id,
        {
          end_date,
          title,
          description,
          category,
          min,
          max,
          rules,
          image: uploadingData.url,
          img: uploadingData,
        },
        { new: true }
      );

      // 🔥 BROADCAST TO ALL WEBSOCKET CLIENTS
      wss.clients.forEach((client) => {
        if (client.readyState === 1) {
          client.send(
            JSON.stringify({
              type: "EDITED_MARKET",
              payload: {
                data: { ...newUser._doc, img: null },
              },
            })
          );
        }
      });

      res.status(201).json({
        data: { ...newUser._doc, img: null },
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request",
      });
    }
  }
);

// DELETE MARKET
Router.delete(
  "/delete/market/:id",
  verifySession,
  verifyAdmin,
  multipartMiddleware,
  lowerCase,
  async (req, res) => {
    try {
      const find_market = await DB.MARKET.findById(req.params.id);

      if (!find_market)
        throw new Error("Unable to complete request, Can't find market");

      if (!find_market.is_resolved)
        throw new Error(
          "Unable to complete request, please resolve market first, before you can now proceed with deleting"
        );

      //   delete the old image here and store the new one
      if (find_market?.img?.public_id) {
        // delete the image HERE
        const deleuploadingData = await cloudinary.uploader.destroy(
          find_market?.img?.public_id
        );

        console.log("uploadingData =>>> ", deleuploadingData);
      }

      const delete_market = await DB.MARKET.findByIdAndDelete(req.params.id);

      res.status(201).json({
        data: { ...delete_market._doc, img: null },
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request",
      });
    }
  }
);

// ADMIN or USER GET MARKETS
Router.get("/get/markets/:page/:limit", lowerCase, async (req, res) => {
  try {
    const page = parseInt(req.params.page) || 1;
    const limit = parseInt(req.params.limit) || 26;
    const uid = req.query.uid; // 👈 you can pass ?uid=USER_ID from frontend

    const total = await DB.MARKET.countDocuments();
    const total_pages = Math.ceil(total / limit);

    const markets = await DB.MARKET.aggregate([
      { $sort: { createdAt: -1 } },
      { $skip: (page - 1) * limit },
      { $limit: limit },

      // Lookup real predictions
      {
        $lookup: {
          from: "predictions",
          let: { marketId: "$_id" },
          pipeline: [
            {
              $match: {
                $expr: {
                  $and: [
                    { $eq: ["$mid", "$$marketId"] },
                    { $eq: ["$is_demo", false] }, // real predictions only
                  ],
                },
              },
            },
          ],
          as: "real_predictions",
        },
      },

      // Lookup demo predictions
      {
        $lookup: {
          from: "predictions",
          let: { marketId: "$_id" },
          pipeline: [
            {
              $match: {
                $expr: {
                  $and: [
                    { $eq: ["$mid", "$$marketId"] },
                    { $eq: ["$is_demo", true] }, // demo predictions only
                  ],
                },
              },
            },
          ],
          as: "demo_predictions",
        },
      },

      // Lookup if the user bookmarked this market
      {
        $lookup: {
          from: "bookmarks",
          let: {
            marketId: "$_id",
            userId: uid ? new mongoose.Types.ObjectId(uid) : null,
          },
          pipeline: [
            {
              $match: {
                $expr: {
                  $and: [
                    { $eq: ["$mid", "$$marketId"] },
                    { $eq: ["$uid", "$$userId"] },
                  ],
                },
              },
            },
            { $limit: 1 },
          ],
          as: "bookmark_info",
        },
      },

      // Add computed stats for both real and demo predictions
      {
        $addFields: {
          isBookmarked: {
            $gt: [{ $size: "$bookmark_info" }, 0],
          },

          // REAL predictions
          total_predictions: { $size: "$real_predictions" },
          yes_predictions: {
            $size: {
              $filter: {
                input: "$real_predictions",
                as: "p",
                cond: { $eq: ["$$p.predict", "yes"] },
              },
            },
          },
          no_predictions: {
            $size: {
              $filter: {
                input: "$real_predictions",
                as: "p",
                cond: { $eq: ["$$p.predict", "no"] },
              },
            },
          },
          total_amount: { $sum: "$real_predictions.amount" },
          yes_amount: {
            $sum: {
              $map: {
                input: {
                  $filter: {
                    input: "$real_predictions",
                    as: "p",
                    cond: { $eq: ["$$p.predict", "yes"] },
                  },
                },
                as: "p",
                in: "$$p.amount",
              },
            },
          },
          no_amount: {
            $sum: {
              $map: {
                input: {
                  $filter: {
                    input: "$real_predictions",
                    as: "p",
                    cond: { $eq: ["$$p.predict", "no"] },
                  },
                },
                as: "p",
                in: "$$p.amount",
              },
            },
          },

          // DEMO predictions
          total_fake_predictions: { $size: "$demo_predictions" },
          yes_fake_predictions: {
            $size: {
              $filter: {
                input: "$demo_predictions",
                as: "p",
                cond: { $eq: ["$$p.predict", "yes"] },
              },
            },
          },
          no_fake_predictions: {
            $size: {
              $filter: {
                input: "$demo_predictions",
                as: "p",
                cond: { $eq: ["$$p.predict", "no"] },
              },
            },
          },
          total_fake_amount: { $sum: "$demo_predictions.amount" },
          yes_fake_amount: {
            $sum: {
              $map: {
                input: {
                  $filter: {
                    input: "$demo_predictions",
                    as: "p",
                    cond: { $eq: ["$$p.predict", "yes"] },
                  },
                },
                as: "p",
                in: "$$p.amount",
              },
            },
          },
          no_fake_amount: {
            $sum: {
              $map: {
                input: {
                  $filter: {
                    input: "$demo_predictions",
                    as: "p",
                    cond: { $eq: ["$$p.predict", "no"] },
                  },
                },
                as: "p",
                in: "$$p.amount",
              },
            },
          },

          // Check if this user has predicted (demo or real)
          hasPredicted: {
            $in: [
              uid ? new mongoose.Types.ObjectId(uid) : null,
              {
                $concatArrays: [
                  "$real_predictions.uid",
                  "$demo_predictions.uid",
                ],
              },
            ],
          },
        },
      },

      // Hide raw prediction arrays and img field
      {
        $project: {
          real_predictions: 0,
          demo_predictions: 0,
          bookmark_info: 0, // 👈 ADD THIS
          img: 0,
        },
      },
    ]);

    const hasMore = page * limit < total;

    res.status(200).json({
      data: {
        data: markets,
        hasMore,
        total_pages,
      },
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      msg: error.message || "Unable to complete request",
    });
  }
});

// ADMIN or USER GET EACH MARKET
Router.get("/each/market/:marketID", lowerCase, async (req, res) => {
  try {
    const uid = req.query.uid; // 👈 you can pass ?uid=USER_ID from frontend
    const id = req.params.marketID;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      throw new Error("Invalid marketID");
    }

    const markets = await DB.MARKET.aggregate([
      { $match: { _id: new mongoose.Types.ObjectId(id) } },

      // Lookup real predictions
      {
        $lookup: {
          from: "predictions",
          let: { marketId: "$_id" },
          pipeline: [
            {
              $match: {
                $expr: {
                  $and: [
                    { $eq: ["$mid", "$$marketId"] },
                    { $eq: ["$is_demo", false] }, // real predictions
                  ],
                },
              },
            },
          ],
          as: "real_predictions",
        },
      },

      // Lookup demo predictions
      {
        $lookup: {
          from: "predictions",
          let: { marketId: "$_id" },
          pipeline: [
            {
              $match: {
                $expr: {
                  $and: [
                    { $eq: ["$mid", "$$marketId"] },
                    { $eq: ["$is_demo", true] }, // demo predictions
                  ],
                },
              },
            },
          ],
          as: "demo_predictions",
        },
      },

      // Lookup if the user bookmarked this market
      {
        $lookup: {
          from: "bookmarks",
          let: {
            marketId: "$_id",
            userId: uid ? new mongoose.Types.ObjectId(uid) : null,
          },
          pipeline: [
            {
              $match: {
                $expr: {
                  $and: [
                    { $eq: ["$mid", "$$marketId"] },
                    { $eq: ["$uid", "$$userId"] },
                  ],
                },
              },
            },
            { $limit: 1 },
          ],
          as: "bookmark_info",
        },
      },

      // Add computed stats for both real and demo predictions
      {
        $addFields: {
          isBookmarked: {
            $gt: [{ $size: "$bookmark_info" }, 0],
          },
          // REAL predictions
          total_predictions: { $size: "$real_predictions" },
          yes_predictions: {
            $size: {
              $filter: {
                input: "$real_predictions",
                as: "p",
                cond: { $eq: ["$$p.predict", "yes"] },
              },
            },
          },
          no_predictions: {
            $size: {
              $filter: {
                input: "$real_predictions",
                as: "p",
                cond: { $eq: ["$$p.predict", "no"] },
              },
            },
          },
          total_amount: { $sum: "$real_predictions.amount" },
          yes_amount: {
            $sum: {
              $map: {
                input: {
                  $filter: {
                    input: "$real_predictions",
                    as: "p",
                    cond: { $eq: ["$$p.predict", "yes"] },
                  },
                },
                as: "p",
                in: "$$p.amount",
              },
            },
          },
          no_amount: {
            $sum: {
              $map: {
                input: {
                  $filter: {
                    input: "$real_predictions",
                    as: "p",
                    cond: { $eq: ["$$p.predict", "no"] },
                  },
                },
                as: "p",
                in: "$$p.amount",
              },
            },
          },

          // DEMO predictions
          total_fake_predictions: { $size: "$demo_predictions" },
          yes_fake_predictions: {
            $size: {
              $filter: {
                input: "$demo_predictions",
                as: "p",
                cond: { $eq: ["$$p.predict", "yes"] },
              },
            },
          },
          no_fake_predictions: {
            $size: {
              $filter: {
                input: "$demo_predictions",
                as: "p",
                cond: { $eq: ["$$p.predict", "no"] },
              },
            },
          },
          total_fake_amount: { $sum: "$demo_predictions.amount" },
          yes_fake_amount: {
            $sum: {
              $map: {
                input: {
                  $filter: {
                    input: "$demo_predictions",
                    as: "p",
                    cond: { $eq: ["$$p.predict", "yes"] },
                  },
                },
                as: "p",
                in: "$$p.amount",
              },
            },
          },
          no_fake_amount: {
            $sum: {
              $map: {
                input: {
                  $filter: {
                    input: "$demo_predictions",
                    as: "p",
                    cond: { $eq: ["$$p.predict", "no"] },
                  },
                },
                as: "p",
                in: "$$p.amount",
              },
            },
          },

          // Check if this user has predicted (real or demo)
          hasPredicted: {
            $in: [
              uid ? new mongoose.Types.ObjectId(uid) : null,
              {
                $concatArrays: [
                  "$real_predictions.uid",
                  "$demo_predictions.uid",
                ],
              },
            ],
          },
        },
      },

      // Hide raw prediction arrays and img
      {
        $project: {
          real_predictions: 0,
          demo_predictions: 0,
          bookmark_info: 0, // 👈 ADD THIS
          img: 0,
        },
      },
    ]);

    res.status(200).json({
      data: markets[0] || null, // single market
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      msg: error.message || "Unable to complete request",
    });
  }
});

const close_market = joi
  .object({
    winning_side: joi.string().valid("yes", "no").required(),
  })
  .strict();

Router.post(
  "/close-market/:marketId",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      const { marketId } = req.params;
      let { error, value } = close_market.validate(req.body);
      if (error) throw new Error(error.details[0].message);

      if (!req.body) {
        return res.status(500).json({ msg: "Request body is missing" });
      }

      const { winning_side } = value; // 'yes' or 'no'

      const market = await DB.MARKET.findById(marketId);
      if (!market) throw new Error("Market not found");
      if (market.is_resolved) throw new Error("Market already resolved");

      const find_app_settings = await DB.APP_SETTINGS.find();

      const predictions = await DB.PREDICTION.find({
        mid: marketId,
        is_demo: false,
      });
      const yesPredictions = predictions.filter((p) => p.predict === "yes");
      const noPredictions = predictions.filter((p) => p.predict === "no");

      const totalYes = yesPredictions.reduce((s, p) => s + p.amount, 0);
      const totalNo = noPredictions.reduce((s, p) => s + p.amount, 0);
      const allTotalVolume = totalYes + totalNo;
      const chooosen_app_per =
        find_app_settings[0]?.app_percentage || APP_PERCENTAGE;
      const app_cut = allTotalVolume * chooosen_app_per;
      const totalVolume = allTotalVolume - app_cut; // remove the app percentage

      const winners = winning_side === "yes" ? yesPredictions : noPredictions;
      const losers = winning_side === "yes" ? noPredictions : yesPredictions;

      const totalWinningVolume = winning_side === "yes" ? totalYes : totalNo;

      const results = [];

      for (const win of winners) {
        const tellUser =
          parseFloat(win.amount / totalWinningVolume) * totalVolume;

        const reward = tellUser - win.amount; // remove the original amount... only remaining the profit

        // Mark prediction resolved
        await DB.PREDICTION.findByIdAndUpdate(win._id, {
          is_resolved: true,
          after_amount: reward,
          winner: "yes",
        });

        results.push({
          userId: win.uid,
          invested: win.amount,
          reward: reward.toFixed(2),
          winner: "yes",
        });

        if (!win.is_seed) {
          // Update wallet: release hold & add reward
          await DB.WALLET.findOneAndUpdate(
            { uid: win.uid },
            {
              $inc: {
                on_hold: -win.amount,
                pending: -win.amount,
                usdt_balance: reward, // add full reward (stake + profit)
                total_balance: reward,
              },
            }
          );

          await DB.USER.findByIdAndUpdate(
            win.uid,
            { $inc: { total_winning: reward } },
            { new: true }
          );

          sendToUser(win.uid?.toString(), {
            type: "ACCOUNT_NOTIFICATION",
            payload: {
              type: "success",
              body: {
                title: "Market",
                text: `Congratulations! Your market prediction proved accurate. Well done!`,
              },
            },
          });

          await DB.NOTIFICATION.create({
            uid: win.uid,
            type: "success",
            body: {
              title: "Market",
              text: `Congratulations! Your market prediction proved accurate. Well done!`,
            },
          });
        }
      }

      // Handle losers: release hold but don't refund
      for (const lose of losers) {
        if (!lose.is_seed) {
          await DB.WALLET.findOneAndUpdate(
            { uid: lose.uid },
            {
              $inc: {
                on_hold: -lose.amount,
                pending: -lose.amount,
                usdt_balance: -lose.amount,
                total_subtracted: lose.amount,
              },
            }
          );
          await DB.PREDICTION.findByIdAndUpdate(lose._id, {
            is_resolved: true,
            after_amount: 0,
            winner: "yes",
          });

          sendToUser(lose.uid?.toString(), {
            type: "ACCOUNT_NOTIFICATION",
            payload: {
              type: "alert",
              body: {
                title: "Market",
                text: `Your market prediction did not align with the outcome this time. Better luck next round.`,
              },
            },
          });

          await DB.NOTIFICATION.create({
            uid: lose.uid,
            type: "alert",
            body: {
              title: "Market",
              text: `Your market prediction did not align with the outcome this time. Better luck next round.`,
            },
          });
        }
      }

      const fakeictions = await DB.PREDICTION.find({
        mid: marketId,
        is_demo: true,
      });
      const yesFakeictions = fakeictions.filter((p) => p.predict === "yes");
      const noFakeictions = fakeictions.filter((p) => p.predict === "no");

      const totalYesFakeictions = yesFakeictions.reduce(
        (s, p) => s + p.amount,
        0
      );
      const totalNoFakeictions = noFakeictions.reduce(
        (s, p) => s + p.amount,
        0
      );
      const allTotalVolumeFakeictions =
        totalYesFakeictions + totalNoFakeictions;
      const totalVolumeFakeictions = allTotalVolumeFakeictions - app_cut; // remove the app percentage

      const winnersFakeictions =
        winning_side === "yes" ? yesFakeictions : noFakeictions;
      const losersFakeictions =
        winning_side === "yes" ? noFakeictions : yesFakeictions;

      const totalWinningVolumeFakeictions =
        winning_side === "yes" ? totalYes : totalNo;

      const resultsFakeictions = [];

      for (const win of winnersFakeictions) {
        const tellUser =
          parseFloat(win.amount / totalWinningVolumeFakeictions) *
          totalVolumeFakeictions;

        const reward = tellUser - win.amount; // remove the original amount... only remaining the profit

        // Mark prediction resolved
        await DB.PREDICTION.findByIdAndUpdate(win._id, {
          is_resolved: true,
          after_amount: reward,
          winner: "yes",
        });

        resultsFakeictions.push({
          userId: win.uid,
          invested: win.amount,
          reward: reward.toFixed(2),
          winner: "yes",
        });

        if (!win.is_seed) {
          // Update wallet: release hold & add reward
          await DB.WALLET.findOneAndUpdate(
            { uid: win.uid },
            {
              $inc: {
                on_hold: -win.amount,
                pending: -win.amount,
                usdt_balance: reward, // add full reward (stake + profit)
                total_balance: reward,
              },
            }
          );

          await DB.USER.findByIdAndUpdate(
            win.uid,
            { $inc: { total_demo_winning: reward } },
            { new: true }
          );

          sendToUser(win.uid?.toString(), {
            type: "ACCOUNT_NOTIFICATION",
            payload: {
              type: "success",
              body: {
                title: "Market",
                text: `Congratulations! Your market prediction proved accurate. Well done!`,
              },
            },
          });

          await DB.NOTIFICATION.create({
            uid: win.uid,
            type: "success",
            body: {
              title: "Market",
              text: `Congratulations! Your market prediction proved accurate. Well done!`,
            },
          });
        }
      }

      // Handle losers: release hold but don't refund
      for (const lose of losersFakeictions) {
        if (!lose.is_seed) {
          await DB.WALLET.findOneAndUpdate(
            { uid: lose.uid },
            {
              $inc: {
                on_hold: -lose.amount,
                pending: -lose.amount,
                usdt_balance: -lose.amount,
                total_subtracted: lose.amount,
              },
            }
          );
          await DB.PREDICTION.findByIdAndUpdate(lose._id, {
            is_resolved: true,
            after_amount: 0,
            winner: "yes",
          });

          sendToUser(lose.uid?.toString(), {
            type: "ACCOUNT_NOTIFICATION",
            payload: {
              type: "alert",
              body: {
                title: "Market",
                text: `Your market prediction did not align with the outcome this time. Better luck next round.`,
              },
            },
          });

          await DB.NOTIFICATION.create({
            uid: lose.uid,
            type: "alert",
            body: {
              title: "Market",
              text: `Your market prediction did not align with the outcome this time. Better luck next round.`,
            },
          });
        }
      }

      market.is_resolved = true;
      market.who_won = winning_side;
      await market.save();

      const commission = await DB.APPCUT.create({
        app_percentage: chooosen_app_per,
        commission: app_cut,
        total_volume: allTotalVolume,
        yes_predictions: yesPredictions.length,
        yes_value: totalYes,
        no_predictions: noPredictions.length,
        no_value: totalNo,
        winning_side,
        mid: market._id,
      });

      res.status(200).json({
        message: `Market closed successfully. ${winning_side.toUpperCase()} won.`,
        totalVolume,
        totalWinningVolume,
        winners: results,
      });
    } catch (error) {
      console.error("Error closing market:", error);
      res.status(500).json({ msg: error.message });
    }
  }
);

module.exports = Router;
