const { DB } = require("../../db");
const { sendToUser } = require("../../middleware/sendToUser");
const Router = require("express").Router();
const { verifyAdmin } = require("../../middleware/verifyAdmin");
const { verifySession } = require("../../middleware/verifySession");
const joi = require("joi");

// admin get metrics data
Router.get("/metrics/data", verifySession, verifyAdmin, async (req, res) => {
  try {
    const total_market = await DB.MARKET.countDocuments();
    const active_users = await DB.USER.countDocuments({ active: true });
    const active_bets = await DB.PREDICTION.countDocuments({
      is_resolved: false,
    });
    const totalvolume = await DB.DE_WITH.aggregate([
      {
        $match: {
          status: "completed",
        },
      },
      {
        $group: {
          _id: {
            type: "$type", // deposit | withdrawal
            method: "$method", // usdt | naira
          },
          totalAmount: { $sum: "$amount" },
        },
      },
    ]);

    // Prepare clean output structure
    const summary = {
      deposit: { usdt: 0, naira: 0 },
      withdrawal: { usdt: 0, naira: 0 },
    };

    totalvolume.forEach((item) => {
      const t = item._id.type; // deposit | withdrawal
      const m = item._id.method; // usdt | naira
      summary[t][m] = item.totalAmount;
    });

    res.status(200).json({
      data: {
        total_market,
        active_users,
        active_bets,
        total_volume: summary,
      },
    });
  } catch (error) {
    console.log("error =>> ", error);

    res.status(500).json({
      msg: error.message || "unable to complete request",
    });
  }
});

// admin getting top markets
Router.get(
  "/top-markets/:page/:limit",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      const page = parseInt(req.params.page) || 1;
      const limit = parseInt(req.params.limit) || 44;

      const skip = (page - 1) * limit;

      // Aggregation pipeline
      const topMarkets = await DB.PREDICTION.aggregate([
        {
          $group: {
            _id: "$mid",
            totalVolume: { $sum: "$amount" },
            totalPredictions: { $sum: 1 },
          },
        },
        { $sort: { totalVolume: -1 } },
        { $skip: skip },
        { $limit: limit },
        {
          $lookup: {
            from: "markets",
            localField: "_id",
            foreignField: "_id",
            as: "market",
          },
        },
        { $unwind: "$market" },
        {
          $project: {
            _id: 0,
            market_id: "$market._id",
            title: "$market.title",
            description: "$market.description",
            category: "$market.category",
            image: "$market.image",
            min: "$market.min",
            max: "$market.max",
            rules: "$market.rules",
            who_won: "$market.who_won",
            is_resolved: "$market.is_resolved",
            end_date: "$market.end_date",
            totalVolume: 1,
            totalPredictions: 1,
          },
        },
      ]);

      // Get total number of grouped markets for pagination
      const countResult = await DB.PREDICTION.aggregate([
        { $group: { _id: "$mid" } },
        { $count: "total" },
      ]);

      const totalMarkets = countResult.length ? countResult[0].total : 0;
      const totalPages = Math.ceil(totalMarkets / limit);

      res.status(200).json({
        data: {
          page,
          limit,
          totalPages,
          totalMarkets,
          data: topMarkets,
        },
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request",
      });
    }
  }
);

// admin wallet metrics
Router.get("/wallet/metrics", verifySession, verifyAdmin, async (req, res) => {
  try {
    const results = await DB.DE_WITH.aggregate([
      {
        $facet: {
          total_naira: [
            { $match: { method: "naira", status: "completed" } },
            {
              $group: {
                _id: null,
                count: { $sum: 1 },
                total: { $sum: "$amount" },
              },
            },
          ],

          total_usdt: [
            { $match: { method: "usdt", status: "completed" } },
            {
              $group: {
                _id: null,
                count: { $sum: 1 },
                total: { $sum: "$amount" },
              },
            },
          ],

          total_deposit: [
            { $match: { type: "deposit", status: "completed" } },
            {
              $group: {
                _id: null,
                count: { $sum: 1 },
                total: { $sum: "$amount" },
              },
            },
          ],

          total_withdrawal: [
            { $match: { type: "withdrawal", status: "completed" } },
            {
              $group: {
                _id: null,
                count: { $sum: 1 },
                total: { $sum: "$amount" },
              },
            },
          ],

          pending_balance: [
            { $match: { status: "pending" } },
            {
              $group: {
                _id: null,
                count: { $sum: 1 },
                total: { $sum: "$amount" },
              },
            },
          ],
        },
      },
    ]);

    const format = (arr) =>
      arr.length
        ? { count: arr[0].count, total: arr[0].total }
        : { count: 0, total: 0 };

    const totalCommission = await DB.APPCUT.aggregate([
      {
        $group: {
          _id: null,
          total: { $sum: "$commission" },
        },
      },
    ]);

    const total_commission = totalCommission[0]?.total || 0;

    res.json({
      data: {
        total_naira: format(results[0].total_naira),
        total_usdt: format(results[0].total_usdt),
        total_deposit: format(results[0].total_deposit),
        total_withdrawal: format(results[0].total_withdrawal),
        pending_balance: format(results[0].pending_balance),
        total_commission,
      },
    });
  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request",
    });
  }
});

// admin get all commissions
Router.get(
  "/all/commissions/:page/:limit",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      const page = parseInt(req.params.page) || 1;
      const limit = parseInt(req.params?.limit) || 26;

      const total = await DB.APPCUT.countDocuments();
      const total_pages = Math.ceil(total / limit);

      const every_ = await DB.APPCUT.find()
        .populate("mid", "-img")
        .skip((page - 1) * limit)
        .limit(limit + 1)
        .sort({ createdAt: -1 });

      const hasMore = every_.length > limit;
      if (hasMore) every_.pop(); // remove the extra item

      res.status(201).json({
        data: {
          data: every_,
          hasMore,
          total_pages,
        },
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request",
      });
    }
  }
);

const SET_SETTINGS = joi
  .object({
    app_percentage: joi.number().required(),
    exchange_rate: joi.number().required(),
    platform_name: joi.string().required(),
    platform_description: joi.string().required(),
    platform_email: joi.string().email().required(),
    support_email: joi.string().email().required(),
    maintenance_mode: joi.boolean().required(),
  })
  .strict();

// admin wallet metrics
Router.put("/edit/settings", verifySession, verifyAdmin, async (req, res) => {
  try {
    let { error, value } = SET_SETTINGS.validate(req.body);
    if (error) throw new Error(error.details[0].message);
    let {
      app_percentage,
      exchange_rate,
      platform_name,
      platform_description,
      platform_email,
      support_email,
      maintenance_mode,
    } = value;

    const findSet = await DB.APP_SETTINGS.find();

    // no settings here... create new settings
    //
    if (findSet.length < 1) {
      const created_ = await DB.APP_SETTINGS.create({
        ...req.body,
      });

      res.json({
        data: created_,
      });
      return;
    }

    const edited_ = await DB.APP_SETTINGS.findByIdAndUpdate(
      findSet[0]._id,
      {
        ...req.body,
      },
      { new: true }
    );

    res.json({
      data: edited_,
    });
  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request",
    });
  }
});

// admin wallet metrics
Router.get("/exchange/rate", async (req, res) => {
  try {
    const findSet = await DB.APP_SETTINGS.find();

    res.json({
      data: findSet[0]?.exchange_rate || 0,
    });
  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request",
    });
  }
});

// getting app details
Router.get("/app/details", async (req, res) => {
  try {
    const findSet = await DB.APP_SETTINGS.find().select(
      "-date -app_percentage -exchange_rate"
    );

    res.json({
      data: findSet[0],
    });
  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request",
    });
  }
});

const SENDMESSSGE = joi
  .object({
    name: joi.string().required(),
    email: joi.string().email().required(),
    subject: joi.string().required(),
    message: joi.string().required(),
  })
  .strict();

// post contact settings
Router.post("/contact/settings", async (req, res) => {
  try {
    let { error, value } = SENDMESSSGE.validate(req.body);
    if (error) throw new Error(error.details[0].message);

    const created_ = await DB.CONTACT.create({
      ...req.body,
    });

    res.json({
      data: created_,
    });
  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request",
    });
  }
});

// get contact/settings
Router.get("/contact/settings", verifySession, async (req, res) => {
  try {
    const created_ = await DB.CONTACT.find().sort({ createdAt: -1 });

    res.json({
      data: created_,
    });
  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request",
    });
  }
});

// delete contact settings
Router.delete(
  "/contact/settings/:ID",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      const created_ = await DB.CONTACT.findByIdAndDelete(req.params.ID);

      res.json({
        data: created_,
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request",
      });
    }
  }
);

module.exports = Router;
