const { DB } = require("../../db");
const { sendToUser } = require("../../middleware/sendToUser");
const Router = require("express").Router();
const { verifyAdmin } = require("../../middleware/verifyAdmin");
const { verifySession } = require("../../middleware/verifySession");
const joi = require("joi");

// admin list all users data
Router.get(
  "/list/users/:page/:limit",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      const page = parseInt(req.params.page) || 1;
      const limit = parseInt(req.params?.limit) || 26;

      const total = await DB.USER.countDocuments();
      const total_pages = Math.ceil(total / limit);

      const totalUsers = total;
      const suspendedUsers = await DB.USER.countDocuments({
        is_suspended: true,
      });
      const activeUsers = await DB.USER.countDocuments({ active: true });
      const notVerifiedUsers = await DB.USER.countDocuments({ is_email_verified: false });

      const bookmarks = await DB.USER.find()
        .select("-img -alert -otp -tfa")
        .skip((page - 1) * limit)
        .limit(limit + 1)
        .sort({ createdAt: -1 });

      const hasMore = bookmarks.length > limit;
      if (hasMore) bookmarks.pop(); // remove the extra item

      res.status(201).json({
        data: {
          metrics: {
            totalUsers,
            suspendedUsers,
            activeUsers,
            notVerifiedUsers
          },
          data: bookmarks,
          hasMore,
          total_pages,
        },
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request",
      });
    }
  }
);

// admin get each user data
Router.get("/each/users/:ID", verifySession, verifyAdmin, async (req, res) => {
  try {
    const bookmarks = await DB.USER.findOne({ _id: req.params.ID }).select(
      "-img -alert -otp -tfa"
    );

    // get demo balance details
    const demoWallet = await DB.WALLET.findOne({
      uid: bookmarks._id,
      is_demo: true,
    });

    const demoBalance = demoWallet
      ? demoWallet.usdt_balance - demoWallet.on_hold
      : 0;

    const mainWallet = await DB.WALLET.findOne({
      uid: bookmarks._id,
      is_demo: true,
    });

    const mainBalance = mainWallet
      ? mainWallet.usdt_balance - mainWallet.on_hold
      : 0;

    res.status(201).json({
      data: {
        data: bookmarks,
        balance: {
          demo: demoBalance,
          main: mainBalance,
        },
      },
    });
  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request",
    });
  }
});

const notification = joi
  .object({
    type: joi.string().valid("alert", "info", "success").required(),
    title: joi.string().required(),
    body: joi.string().required(),
  })
  .strict();

// admin send notification msg with websocket
Router.post(
  "/send/notification/:ID",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      let { error, value } = notification.validate(req.body);
      if (error) throw new Error(error.details[0].message);

      const { title, body, type } = value;

      const findSession = await DB.USER.findById(req.params.ID);
      if (!findSession) throw new Error("Can't find user with the required ID");

      sendToUser(findSession._id?.toString(), {
        type: "ACCOUNT_NOTIFICATION",
        payload: {
          type,
          body: {
            title,
            text: body,
          },
        },
      });

      await DB.NOTIFICATION.create({
        uid: findSession._id,
        type,
        body: {
          title,
          text: body,
        },
      });

      res.status(201).json({
        data: "notification sent",
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request",
      });
    }
  }
);

// admin toggle/suspension for any user
Router.put(
  "/toggle/suspend/:ID",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      const findSession = await DB.USER.findById(req.params.ID);
      if (!findSession) throw new Error("Can't find user with the required ID");

      const currentSus = findSession.is_suspended;

      await DB.USER.findByIdAndUpdate(
        findSession._id,
        { is_suspended: !currentSus },
        { new: true }
      );

      res.status(201).json({
        data: currentSus
          ? `${findSession.username} has been reinstated`
          : `${findSession.username} has been suspended`,
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request",
      });
    }
  }
);

module.exports = Router;
